import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import 'package:razinsoft_rider/core/state/app_state.dart';
import 'package:razinsoft_rider/data/models/order_response/order_model/order/order.dart';
import 'package:razinsoft_rider/data/models/waypoint.dart';
import 'package:razinsoft_rider/data/services/local_storage_service.dart';
import 'package:razinsoft_rider/data/services/navigation_service.dart';
import 'package:razinsoft_rider/presentation/booking/provider/order_providers.dart';
import '../../../core/routes/app_routes.dart';
import '../../../core/utils/helpers.dart';
import '../../../data/repositories/interfaces/order_repo_interface.dart';
import '../../splash/provider/app_flow_provider.dart';
import '../../track_order/provider/track_order_provider.dart';
import '../provider/booking_providers.dart';
import '../provider/pusher_provider.dart';

class CreateOrderNotifier extends StateNotifier<AppState<Order>> {
  final IOrderRepo orderRepo;
  final Ref ref;

  CreateOrderNotifier(this.orderRepo, this.ref)
      : super(const AppState.initial());

  Future<void> createOrder({required Map<String, dynamic> orderData}) async {
    state = const AppState.loading();
    final result = await orderRepo.createOrder(data: orderData);
    result.fold(
          (failure) => _handleFailure(failure),
          (data) => handleOrderCreationSuccess(data.data!.order!,),
    );
  }

  Future<void> orderDetailsForCancelRide({required int orderId,}) async {
    state = const AppState.loading();
    final result = await orderRepo.orderDetails(orderId: orderId);

    result.fold(
          (failure) => state = AppState.error(failure),
          (data) async {
        state = AppState.success(data.data!);
      },
    );
  }

  Future<void> setOrderData(Order order)async{
    await LocalStorageService().saveOrderId(order.id);
    state = AppState.success(order);

  }

  void reset() {
    state = const AppState.initial();
  }

  Future<void> handleOrderCreationSuccess(Order data, ) async {
    state = AppState.success(data);
    await LocalStorageService().saveOrderId(data.id);
    await ref.read(pusherNotifierProvider.notifier).setupPusherListeners(connectAllChannels: true);

    ref.read(trackOrderNotifierProvider.notifier).reset();
    ref.read(bookingNotifierProvider.notifier)
    ..resetState()
    ..inProgress();

  }

  List<Waypoint> constructWaypoints(Order data) {
    final waypoints = <Waypoint>[
      Waypoint(
        name: 'Pick-up point',
        address: data.addresses?.pickupAddress ?? '',
        location: LatLng(
          data.points?.pickupLocation?.first.toDouble() ?? 0,
          data.points?.pickupLocation?.last.toDouble() ?? 0,
        ),
      ),
      Waypoint(
        name: 'Drop-off Point',
        address: data.addresses?.dropAddress ?? '',
        location: LatLng(
          data.points?.dropLocation?.first.toDouble() ?? 0,
          data.points?.dropLocation?.last.toDouble() ?? 0,
        ),
      ),
    ];

    return waypoints;
  }

  void navigateToHome({String? status, bool? isSuccess = true}) {
    ref.read(pusherNotifierProvider.notifier).disconnect();
    LocalStorageService().clearOrderId();
    showNotification( message: 'Your Ride was $status', isSuccess: isSuccess ?? false);
    NavigationService.pushNamedAndRemoveUntil(AppRoutes.dashboard,);
  }

  void _handleFailure(failure) {
    state = AppState.error(failure);
    showNotification(
      message: failure.message,
    );
  }
}

class CheckTripActivityNotifier extends StateNotifier<AppState<Order?>> {
  final IOrderRepo orderRepo;
  final Ref ref;

  CheckTripActivityNotifier(this.orderRepo, this.ref)
      : super(const AppState.initial());
  Future<void> checkTripActivity() async {
    if(await LocalStorageService().getToken() != null){
      state = const AppState.loading();
      final result = await orderRepo.checkActiveTrip();
      result.fold(
            (failure) => _handleFailure(failure),
            (data) {
          if(data.data?.order != null){
            state = AppState.success(data.data!.order);
            ref.read(createOrderNotifierProvider.notifier).setOrderData(data.data!.order!);
          }else{
            state = const AppState.success(null);
          }
        },
      );
    }else{
      state = const AppState.success(null);
    }
    ref.read(appFlowNotifierProvider.notifier).setAppFlowState();
  }

  void _handleFailure(failure) {
    state = AppState.error(failure);
    showNotification(
      message: failure.message,
    );
  }

}
